/*
 * Decompiled with CFR 0.152.
 */
package jayeson.lib.streamfinder;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.collect.ImmutableMap;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.google.inject.name.Named;
import java.io.IOException;
import java.net.ConnectException;
import java.net.URI;
import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.function.Function;
import java.util.stream.Collectors;
import jayeson.lib.streamfinder.AuthenticationException;
import jayeson.lib.streamfinder.SessionException;
import jayeson.lib.streamfinder.SessionTicket;
import jayeson.lib.streamfinder.SessionToken;
import jayeson.lib.streamfinder.UserCredential;
import jayeson.lib.streamfinder.internal.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import play.libs.ws.StandaloneWSClient;
import play.libs.ws.StandaloneWSRequest;
import play.libs.ws.StandaloneWSResponse;

@Singleton
public class SessionFactory {
    Duration retryDelay;
    final Utils util;
    final StandaloneWSClient httpClient;
    final Map<UserCredential, CompletionStage<SessionToken>> sessions;
    static final Duration HTTP_TIMEOUT_MS = Duration.ofMillis(5000L);
    static final Logger log = LoggerFactory.getLogger(SessionFactory.class);
    static final ObjectMapper mapper = new ObjectMapper();

    @Inject
    public SessionFactory(@Named(value="SF_HTTP_CLIENT") StandaloneWSClient httpClient, Utils util) {
        this.httpClient = httpClient;
        this.sessions = new HashMap<UserCredential, CompletionStage<SessionToken>>();
        this.retryDelay = HTTP_TIMEOUT_MS;
        this.util = util;
    }

    public synchronized CompletionStage<SessionToken> getSession(String username, String password, String url) {
        if (username == null) {
            throw new IllegalArgumentException("Username cannot be null");
        }
        if (password == null) {
            throw new IllegalArgumentException("Password cannot be null");
        }
        URI endpoint = Utils.parseUri(url);
        UserCredential user = new UserCredential(username, password, endpoint);
        CompletionStage session = this.sessions.computeIfAbsent(user, this::doLogin);
        return session.whenComplete((token, error) -> this.purgeUnauthToken((SessionToken)token, (Throwable)error, user));
    }

    public synchronized boolean hasSession(String username, String password, String url) {
        if (username == null) {
            throw new IllegalArgumentException("Username cannot be null");
        }
        if (password == null) {
            throw new IllegalArgumentException("Password cannot be null");
        }
        URI endpoint = Utils.parseUri(url);
        UserCredential user = new UserCredential(username, password, endpoint);
        return this.sessions.containsKey(user);
    }

    CompletionStage<SessionToken> doLogin(UserCredential user) {
        log.debug("Logging in " + user.getUsername() + ", " + user.getPassword() + ", " + user.getUrl());
        StandaloneWSRequest query = this.httpClient.url(user.getUrl().toString() + "/auth/login").setRequestTimeout(HTTP_TIMEOUT_MS);
        ImmutableMap loginParams = ImmutableMap.of((Object)"username", (Object)user.getUsername(), (Object)"accessToken", (Object)user.getPassword());
        CompletionStage<StandaloneWSResponse> request = Utils.post(query, loginParams);
        return request.thenApply(SessionFactory::parseLoginResponse);
    }

    SessionToken purgeUnauthToken(SessionToken token, Throwable error, UserCredential user) {
        if (error != null) {
            this.invalidate(user.getUsername(), user.getPassword(), user.getUrl().toString());
        } else if (token != null && !token.isAuthenticated()) {
            this.invalidate(user.getUsername(), user.getPassword(), user.getUrl().toString());
        }
        return token;
    }

    public synchronized void invalidate(String username, String password, String url) {
        if (username == null) {
            throw new IllegalArgumentException("Username cannot be null");
        }
        if (password == null) {
            throw new IllegalArgumentException("Password cannot be null");
        }
        URI endpoint = Utils.parseUri(url);
        this.sessions.remove(new UserCredential(username, password, endpoint));
    }

    @Deprecated
    public synchronized CompletionStage<String> makeTicket(String username, String password, String url, String clientId) {
        if (username == null) {
            throw new IllegalArgumentException("Username cannot be null");
        }
        if (password == null) {
            throw new IllegalArgumentException("Password cannot be null");
        }
        if (clientId == null) {
            throw new IllegalArgumentException("ClientId cannot be null");
        }
        if (!this.hasSession(username, password, url)) {
            throw new AuthenticationException("No session found for " + username + ". Call getSession first");
        }
        URI endpoint = Utils.parseUri(url);
        CompletionStage<SessionToken> getSession = this.sessions.get(new UserCredential(username, password, endpoint));
        return getSession.thenCompose(token -> {
            if (!token.getError().isEmpty()) {
                throw new AuthenticationException(token.getError());
            }
            return this.sendMakeTicket((SessionToken)token, url, clientId);
        }).thenApply(SessionTicket::getTicket);
    }

    public CompletionStage<SessionTicket> makeAuthContent(String username, String password, String url, String clientId) {
        if (clientId == null) {
            throw new IllegalArgumentException("ClientId cannot be null");
        }
        return this.loopQueryTicket(username, password, url, clientId);
    }

    public Duration getRetryDelay() {
        return this.retryDelay;
    }

    public void setRetryDelay(Duration retryDelay) {
        this.retryDelay = retryDelay;
    }

    CompletionStage<SessionTicket> loopQueryTicket(String username, String password, String url, String clientId) {
        return this.loopQuerySession(username, password, url).thenCompose(token -> this.sendMakeTicket((SessionToken)token, url, clientId)).thenCompose(SessionFactory::checkTicket).handle(SessionFactory::handleNetworkError).thenCompose(Function.identity()).thenCompose(ticket -> {
            if (ticket == null) {
                this.invalidate(username, password, url);
                return this.util.waitFor(this.retryDelay).thenCompose(nil -> this.loopQueryTicket(username, password, url, clientId));
            }
            return CompletableFuture.completedFuture(ticket);
        });
    }

    static CompletionStage<SessionTicket> checkTicket(SessionTicket ticket) {
        if (ticket.getTicket().isEmpty()) {
            CompletableFuture<SessionTicket> check = new CompletableFuture<SessionTicket>();
            check.completeExceptionally(new SessionException("Cannot make ticket"));
            return check;
        }
        return CompletableFuture.completedFuture(ticket);
    }

    CompletionStage<SessionToken> loopQuerySession(String username, String password, String url) {
        return this.getSession(username, password, url).handle(SessionFactory::handleNetworkError).thenCompose(Function.identity()).thenCompose(token -> {
            if (token == null) {
                return this.util.waitFor(this.retryDelay).thenCompose(nil -> this.loopQuerySession(username, password, url));
            }
            return CompletableFuture.completedFuture(token);
        });
    }

    static <T> CompletionStage<T> handleNetworkError(T in, Throwable error) {
        if (in != null) {
            return CompletableFuture.completedFuture(in);
        }
        try {
            Throwable c = error.getCause();
            if (c != null) {
                throw c;
            }
            throw error;
        }
        catch (ConnectException | SessionException e) {
            return CompletableFuture.completedFuture(null);
        }
        catch (Throwable e) {
            CompletableFuture future = new CompletableFuture();
            future.completeExceptionally(e);
            return future;
        }
    }

    static SessionToken parseLoginResponse(StandaloneWSResponse httpResponse) {
        if (httpResponse == null) {
            return new SessionToken("", "", "", "");
        }
        try {
            JsonNode response = mapper.readTree(httpResponse.getBody());
            boolean isAuthenticated = response.path("status").asText().equals("SUCCESS");
            String cookieHeader = SessionFactory.parseCookies(httpResponse);
            if (isAuthenticated) {
                return new SessionToken(response.path("sessionId").asText(), response.path("accessToken").asText(), cookieHeader, "");
            }
            throw new AuthenticationException("Cannot authenticate user. Check username and password");
        }
        catch (IOException e) {
            log.error("Exception reading httpResponse {}", (Throwable)e);
            return new SessionToken("", "", "", "");
        }
    }

    static String parseCookies(StandaloneWSResponse httpResponse) {
        List cookies = httpResponse.getCookies();
        return cookies.stream().map(cookie -> cookie.getName() + "=" + cookie.getValue()).collect(Collectors.joining(";"));
    }

    CompletionStage<SessionTicket> sendMakeTicket(SessionToken token, String url, String clientId) {
        JsonNode params = SessionFactory.makeTicketRequest(clientId);
        CompletionStage<StandaloneWSResponse> request = Utils.post(this.httpClient.url(url + "/auth/makeTicket").setRequestTimeout(HTTP_TIMEOUT_MS).addHeader("Cookie", token.getCookie()), params);
        return request.thenApply(response -> SessionFactory.parseTicketResponse(response, token));
    }

    static SessionTicket parseTicketResponse(StandaloneWSResponse httpResponse, SessionToken token) {
        if (httpResponse.getStatus() == 401) {
            throw new SessionException("Invalid session token");
        }
        try {
            JsonNode response = mapper.readTree(httpResponse.getBody());
            JsonNode ticketField = response.path("ticket");
            if (!ticketField.isMissingNode()) {
                return new SessionTicket(token, ticketField.asText());
            }
            log.error("Unable to parse ticket response " + response);
            return new SessionTicket(token, "");
        }
        catch (IOException e) {
            log.error("Exception reading httpResponse {}", (Throwable)e);
            return new SessionTicket(token, "");
        }
    }

    static JsonNode makeTicketRequest(String clientId) {
        ObjectNode root = JsonNodeFactory.instance.objectNode();
        root.put("clientId", clientId);
        return root;
    }
}

