<?php

use JsonStreamingParser\Parser;
use JsonStreamingParser\Listener\ListenerInterface;

require_once 'vendor/autoload.php'; // Adjust path as needed

class MultiJsonListener implements ListenerInterface
{
    private $stack = [];
    private $keyStack = [];
    private $currentKey = null;
    private $depth = 0;
    private $onJson;

    public function __construct(callable $onJson)
    {
        $this->onJson = $onJson;
    }

    public function startDocument(): void {}

    public function endDocument(): void {}

    public function startObject(): void
    {
        $this->stack[] = [];
        $this->keyStack[] = $this->currentKey;
        $this->currentKey = null;
        $this->depth++;
    }

    public function endObject(): void
    {
        $obj = array_pop($this->stack);
        $key = array_pop($this->keyStack);
        $this->depth--;

        if (empty($this->stack)) {
            // Top-level object → emit
            call_user_func($this->onJson, $obj);
        } else {
            $this->addValue($obj, $key);
        }
    }

    public function startArray(): void
    {
        $this->stack[] = [];
        $this->keyStack[] = $this->currentKey;
        $this->currentKey = null;
        $this->depth++;
    }

    public function endArray(): void
    {
        $arr = array_pop($this->stack);
        $key = array_pop($this->keyStack);
        $this->depth--;

        $this->addValue($arr, $key);
    }

    public function key(string $key): void
    {
        $this->currentKey = $key;
    }

    public function value($value): void
    {
        $key = $this->currentKey;
        $this->currentKey = null;
        $this->addValue($value, $key);
    }

    public function whitespace(string $whitespace): void {}

    private function addValue($value, $key): void
    {
        $containerIndex = count($this->stack) - 1;
        $container = &$this->stack[$containerIndex];

        if (is_array($container)) {
            if ($key === null) {
                // Array element
                $container[] = $value;
            } else {
                // Object property
                $container[$key] = $value;
            }
        }
    }
}

function sendSubscribeMessage($socket): void
{
    $subscribeMessage = '{"type":"SUBSCRIBE"}';
    echo "Sending subscribe message...\n";
    fwrite($socket, $subscribeMessage);
}

function processMessage(array $message): void
{
    echo "Received message:\n";
    echo json_encode($message, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE) . "\n\n";
}

function readMessageStream($socket): void
{
    $listener = new MultiJsonListener('processMessage');
    $parser = new Parser($socket, $listener);

    try {
        $parser->parse();
    } catch (Exception $e) {
        fwrite(STDERR, "Parsing error: {$e->getMessage()}\n");
    }
}

// ===== Main Program =====

$host = 'localhost';
$port = 8992;

echo "Connecting to $host:$port\n";

$socket = @stream_socket_client("tcp://$host:$port", $errno, $errstr, 5);
if (!$socket) {
    fwrite(STDERR, "Connection failed: $errstr ($errno)\n");
    exit(1);
}

echo "Connected\n";

sendSubscribeMessage($socket);

readMessageStream($socket);

fclose($socket);
