package deltafeedapp.subscribers;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;

import deltafeedapp.pushmodehandlers.SoccerMatchPushModeHandler;
import jayeson.lib.feed.api.IBetEvent;
import jayeson.lib.feed.api.IBetMatch;
import jayeson.lib.feed.api.IBetRecord;
import jayeson.lib.feed.api.ISnapshot;
import jayeson.lib.feed.api.SportType;
import jayeson.lib.feed.api.twoside.IB2Match;
import jayeson.lib.feed.api.twoside.IB2Record;
import jayeson.lib.feed.soccer.SoccerMatch;
import jayeson.lib.sports.client.FeedView;
import jayeson.lib.sports.client.SportsFeedClient;
import jayeson.lib.sports.client.SportsFeedFactory;
import jayeson.lib.sports.codec.FilterData;
import jayeson.model.filterrules.SportFilterRule;

public class UpstreamFilterSportSubscriber {

	public static void main(String[] args) throws InterruptedException {	
		/* Creating factory before instantiating SportsFeedClient */
		SportsFeedFactory factory = new SportsFeedFactory();
		
		/* Create SportsFeedClient using default config file (located in conf folder - libSportsConfig.json) */
		SportsFeedClient client = factory.create();
		
		/* Start the client */
		client.start();

		/* You can also create a view with specific sport type to use sport specific api */
		FeedView<SoccerMatch> soccerMatchFeedView = client.view(SoccerMatch.class);
		
		/* Make sure you implement at either upstream or local filter to filter out other sport types 
		 * as the FeedView declared above only support SoccerMatch as what you have parameterized
		/* This example retrieves only soccer odds. */
		SportFilterRule sportRule = new SportFilterRule();
		sportRule.setSportTypes(Arrays.asList(SportType.SOCCER));

		/* Assign the specified filters into the FilterData object and start using them */
		FilterData soccerFilterData = new FilterData("default", Arrays.asList(sportRule));
		/* Server side filter decides the records to be received 
		   Pass in the FilterData object into this function to set an upstream filter */
		client.setUpStreamFilter(soccerFilterData);
		
		/* You can now retrieve data from the newly created soccer view. You can choose to process this data by polling the view or by attaching an event handler. */
		/* Process events by attaching an event handler */
		SoccerMatchPushModeHandler myHandler = new SoccerMatchPushModeHandler("UpstreamFilterBySoccer");
		soccerMatchFeedView.register(myHandler);

		/* Read data by polling the view from a feed view, basically it is to repeatedly poll for the latest snapshot */
		while (true) {
			int limit = 3;
			ISnapshot<SoccerMatch> snapshot = soccerMatchFeedView.snapshot();
		    Collection<SoccerMatch> matches = snapshot.matches();
		    System.out.print("There are currently " + matches.size() + " matches. ");
		    System.out.print("Only printing "+limit+" matches/events/records\n");
		    printMatches(matches,limit);
		    /* Perform business logic on the list of matches */
		    Thread.sleep(5000);
		}
		
		/* Once an EventHandler is no longer needed, it should be unregistered from a view */
//		filterView.unregister(myHandler);
//		client.removeView(filterView);
		
	}
	/* Below are printing function for display purposes */
	public static <M extends IBetMatch> void printMatches(Collection<M> matches, int limit) {
	    Iterator<M> it = matches.iterator();
	    for (int i = 0; i < limit && it.hasNext(); i++) {
	        M match = it.next();
	        System.out.println("-------------------------");
	        System.out.println(match.sportType() + ":" + match.league() + ":" + ((IB2Match)match).participantOne() + ":" + ((IB2Match)match).participantTwo());
	        printEvents(match.events(), limit);
	    }
	}
	
	public static <E extends IBetEvent> void printEvents(Collection<E> events, int limit) {
        Iterator<E> it = events.iterator();
        for (int i = 0; i < limit && it.hasNext(); i++) {
        	IBetEvent event = it.next();
            printRecords(event.records(), limit);
        }
	}
	
	public static <R extends IBetRecord> void printRecords(Collection<R> records, int limit) {
	    Iterator<R> it = records.iterator();
        for (int i = 0; i < limit && it.hasNext(); i++) {
        	IBetRecord record = it.next();
            System.out.println(record.source() + ":" + ((IB2Record)record).pivotType() + ":" + ((IB2Record)record).oddType() + ":" + ((IB2Record)record).rateOver() + ":" + ((IB2Record)record).rateUnder() + ":" + ((IB2Record)record).rateEqual());
        }
	}
	
	public static <M extends IBetMatch> void printMatches(Collection<M> matches) {
		printMatches(matches, matches.size());
	}
	
	public static <E extends IBetEvent> void printEvents(Collection<E> events) {
		printEvents(events, events.size());
	}
	
	public static <R extends IBetRecord> void printRecords(Collection<R> records) {
		printRecords(records, records.size());
	}
}